// -----------------------------------------------------------------------------
// Project:	TToolbarEditor
// Module:	tbedsgn - Registration and Design time toolbar editor
// -----------------------------------------------------------------------------
unit tbedsgn;

// -----------------------------------------------------------------------------
//
//			Interface
//
// -----------------------------------------------------------------------------
interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  DsgnIntf, DsgnWnds, Buttons, Menus, ComCtrls, StdCtrls, ExtCtrls,
  tbedit;

type
  TToolbarComponentEditor = class(TComponentEditor)
    private
      procedure DesignWndProc(var msg: TMessage);
      procedure MouseMove(Shift: TShiftState; X, Y: Integer);
      procedure MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
      procedure AnimateLine(Sender: TObject);
      procedure DrawLine(NewPos: TPoint; DrawNew: Boolean);
      procedure SelectControl(Control: TControl; SelectNew: Boolean);

    protected
      procedure ShowToolbarDesigner(Component: TToolbarEditor);
      procedure SelectToolbar(Component: TToolbarEditor);
      procedure About;

    public
      procedure ExecuteVerb(Index: Integer); override;
      function GetVerb(Index: Integer): string; override;
      function GetVerbCount: Integer; override;
  end;

  TToolbarCategoryEditor = Class(TStringProperty)
    public
      function  GetAttributes : TPropertyAttributes; override;
      procedure GetValues(Proc: TGetStrProc); override;
  end;

  TFormDesignToolEdit = class(TDesignWindow)
    PanelTree: TPanel;
    PanelPreview: TPanel;
    GroupBoxPreview: TGroupBox;
    SpeedButtonPreview: TSpeedButton;
    LabePreviewlDescription: TLabel;
    ListViewButtons: TListView;
    ImageListButtons: TImageList;
    PopupMenu: TPopupMenu;
    MenuItemImport: TMenuItem;
    MenuItemRemove: TMenuItem;
    MenuItemRemoveAll: TMenuItem;
    N1: TMenuItem;
    MenuItemDone: TMenuItem;
    N2: TMenuItem;
    MenuItemView: TMenuItem;
    MenuItemViewList: TMenuItem;
    MenuItemViewIcons: TMenuItem;
    procedure ButtonOKClick(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure FormDestroy(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure ButtonImportClick(Sender: TObject);
    procedure ListViewButtonsChange(Sender: TObject; Item: TListItem;
      Change: TItemChange);
    procedure ButtonClearClick(Sender: TObject);
    procedure ListViewButtonsColumnClick(Sender: TObject;
      Column: TListColumn);
    procedure FormCreate(Sender: TObject);
    procedure ListViewButtonsKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure MenuItemRemoveClick(Sender: TObject);
    procedure FormKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure MenuItemViewListClick(Sender: TObject);
    procedure MenuItemViewIconsClick(Sender: TObject);
  private
    { Private declarations }
    SortColumn: Integer;
    ListDirty: Boolean;
    procedure SortList;
    procedure UpdatePreview;
    procedure BuildList;
    procedure UpdateList;
    procedure UpdateSelection;
  protected
    procedure Activated; override;
  public
    { Public declarations }
    Component: TToolbarEditor;

    procedure FormClosed(AForm: TForm); override;
    procedure ComponentDeleted(DelComponent: TComponent); override;
    procedure FormModified; override;
    procedure SelectionChanged(ASelection: TComponentList); override;
  end;

  procedure Register;

// -----------------------------------------------------------------------------
//
//			Implementation
//
// -----------------------------------------------------------------------------
implementation

// Design time resources
// This includes all forms used at design time.
// Note: Resources have been moved to *.DCR file!
//$R *.RES}

uses
  ExptIntf,
  registry,
  libmain,
  tbeadd;

// -----------------------------------------------------------------------------
// Visual method(s) used for design time toolbar selection
type
  TSelectTypes = (stInvert,	// Invert selected area
  		stFrame,	// Draw thick focus rect around selection
  		stCursor);	// Change cursor shape to reflect selection
  TSelectType = set of TSelectTypes;

const
  DesignSelectType = [stCursor, stFrame];

const
  LineMask: Word = $E0E0; // Bitmapped line pattern: 1110000011100000 ( 1 = transparent )
  LineSpeed = 3; // Animation speed

// -----------------------------------------------------------------------------
//
//			Registration
//
// -----------------------------------------------------------------------------
procedure Register;
begin
  RegisterComponents('HCU', [TToolbarEditor]);
  RegisterNoIcon([TToolbarItem]);
  RegisterPropertyEditor(TypeInfo(String), TToolbarItem, 'Category', TToolbarCategoryEditor);
  RegisterComponentEditor(TToolbarEditor, TToolbarComponentEditor);
end;

// -----------------------------------------------------------------------------
//
//			TToolbarComponentEditor
//
// -----------------------------------------------------------------------------
procedure TToolbarComponentEditor.ExecuteVerb(Index: Integer);
begin
  case Index of
    0: ShowToolbarDesigner(TToolbarEditor(Component));
    1: SelectToolbar(TToolbarEditor(Component));
    2: About;
  end;
end;

// -----------------------------------------------------------------------------
function TToolbarComponentEditor.GetVerb(Index: Integer): string;
begin
  case Index of
    0: Result := 'Edit';
    1: Result := 'Connect Toolbar';
    2: Result := 'About...';
  end;
end;

// -----------------------------------------------------------------------------
function TToolbarComponentEditor.GetVerbCount: Integer;
begin
  Result := 3;
end;

// -----------------------------------------------------------------------------
procedure TToolbarComponentEditor.ShowToolbarDesigner(Component: TToolbarEditor);
var
  FormDesignToolEdit: TFormDesignToolEdit;
begin
  if (Component.DesignTimeDesigner = nil) then
  begin
    FormDesignToolEdit := TFormDesignToolEdit.create(Application);
    FormDesignToolEdit.Designer := Designer;
    FormDesignToolEdit.Component := Component;
    Component.DesignTimeDesigner := FormDesignToolEdit;
  end;

  Component.DesignTimeDesigner.Show;
end;

// -----------------------------------------------------------------------------
type
  TFormTBEAbout = class(TForm)
    ButtonOK: TButton;
    LabelText: TLabel;
    PanelGlyph: TPanel;
    PanelClip: TPanel;
    SpeedButtonGlyph: TSpeedButton;
  end;

procedure TToolbarComponentEditor.About;
var
  FormAbout		: TFormTBEAbout;
begin
  FormAbout:= TFormTBEAbout.Create(Application);
  try
    FormAbout.LabelText.Caption := 'Toolbar Editor v' + ToolbarEditorVersion + #13
      + 'by Anders Melander' + #13
      + 'anme@biocat.ruc.dk';
    FormAbout.ShowModal;
  finally
    FormAbout.Release;
  end;
end;

// -----------------------------------------------------------------------------
// Select Toolbar visually
var
  LastTarget		: TControl;
const
  IsGrabbing: Boolean = False;
  CaptureWindow: THandle = 0;


procedure TToolbarComponentEditor.SelectToolbar(Component: TToolbarEditor);
var
  StartPoint		: TPoint;
  Rect			: TRect;
begin
  if (CaptureWindow = 0) then
    CaptureWindow := AllocateHWnd(DesignWndProc);

  Designer.Form.Canvas.Pen.Color := clWhite;
  Designer.Form.Canvas.Pen.Mode := pmXor;
  Designer.Form.Canvas.Pen.Style := psSolid;
  Designer.Form.Canvas.Pen.Width := 1;

  // Initial selection
  if (stCursor in DesignSelectType) then
    Screen.Cursor := crNo;

  // Draw first line ( CompIconSize = Size of on-form component image defined in LibMain )
  StartPoint.X := LongRec(Component.DesignInfo).lo + (CompIconSize DIV 2);
  StartPoint.Y := LongRec(Component.DesignInfo).hi + (CompIconSize DIV 2);
  StartPoint := Designer.Form.ClientToScreen(StartPoint);
  DrawLine(StartPoint, True);
  LastTarget := nil;

  // Capture mouse
  SetCapture(CaptureWindow);

  // Confine cursor
  Rect.TopLeft := Designer.Form.ClientOrigin;
  Rect.BottomRight := Point(Rect.TopLeft.X + Designer.Form.ClientWidth,
    Rect.TopLeft.Y + Designer.Form.ClientHeight);

  ClipCursor(@Rect);
end;

// -----------------------------------------------------------------------------
// Design time windows message dispatcher
procedure TToolbarComponentEditor.DesignWndProc(var msg: TMessage);
const
  isInside: Boolean = False;

  procedure WMMouseMove(var Message: TWMMouseMove);
  begin
    MouseMove(KeysToShiftState(Message.Keys), Message.XPos, Message.YPos);
  end;

  procedure WMLButtonUp(var Message: TWMLButtonUp);
  begin
    MouseUp(mbLeft, KeysToShiftState(Message.Keys), Message.XPos, Message.YPos);
  end;
begin
  if (isInside) then
    exit;
  isInside:= True;
  try
    case (msg.msg) of
      WM_MOUSEMOVE:
        WMMouseMove(TWMMouseMove(msg));
      WM_LBUTTONDOWN,
      WM_LBUTTONUP:
        WMLButtonUp(TWMLButtonUp(msg));
    end;
  finally
    isInside:= False;
  end;
end;


// -----------------------------------------------------------------------------
var
  WorkMask		: Word; // Line pattern work value

procedure TToolbarComponentEditor.AnimateLine(Sender: TObject);
var
  Pos			: TPoint;
begin
  GetCursorPos(Pos);
  DrawLine(Pos, True);
end;

procedure TToolbarComponentEditor.DrawLine(NewPos: TPoint; DrawNew: Boolean);
const
  LineDrawn		: Boolean = False;
  FromPos		: TPoint = ();
  LastPos		: TPoint = ();
  LastMask		: Word = 0;
  AnimateTimer		: TTimer = nil;

  procedure DrawAnimatedPixel(X,Y: Integer; Canvas: TCanvas); stdcall;
  begin
    // Draw line pixel and rotate pattern
    if (WorkMask and 1 = 0) then
    begin
      Canvas.Pixels[X,Y] := clSilver;
      WorkMask := (WorkMask shr 1);
    end else
    begin
      Canvas.Pixels[X,Y] := clBlack;
      WorkMask := (WorkMask shr 1) or $8000;
    end;
  end;

  procedure DrawHandle(Pos: TPoint);
  begin
    Designer.Form.Canvas.Pen.Width := 2;
    Designer.Form.Canvas.Rectangle(Pos.X-2,Pos.Y-2,Pos.X+2,Pos.Y+2);
    Designer.Form.Canvas.Pen.Width := 1;
  end;

begin
  // Rotate line pattern
  if (DrawNew) then
    LineMask := (LineMask shr LineSpeed) or (LineMask shl (16-LineSpeed))
  else
  begin
    AnimateTimer.Free;
    AnimateTimer := nil;
  end;

  // Erase old line
  if (LineDrawn) then
  begin
    // If rectangle hasn't moved, only erase difference (to avoid flicker)
    if (NewPos.X = LastPos.X) and (NewPos.Y = LastPos.Y) and DrawNew then
    begin
      WorkMask := not(LineMask XOR LastMask);
      LastMask := LineMask;
    end else
    begin
      WorkMask := LastMask;
      LineDrawn := False;
    end;
    DrawHandle(FromPos);
    LineDDA(LastPos.X,LastPos.Y,FromPos.X,FromPos.Y,
      @DrawAnimatedPixel,LongInt(Designer.Form.Canvas));
    DrawHandle(LastPos);
  end else
  begin
    // First time called
    FromPos := Designer.Form.ScreenToClient(NewPos);
  end;

  if (DrawNew) and not(LineDrawn) then
  begin
    if (AnimateTimer = nil) then
    begin
      AnimateTimer:= TTimer.Create(Component);
      AnimateTimer.Interval := 100;
      AnimateTimer.Enabled := True;
      AnimateTimer.OnTimer := AnimateLine;
    end;
    // Draw new line
    NewPos := Designer.Form.ScreenToClient(NewPos);
    DrawHandle(FromPos);
    WorkMask := LineMask;
    LineDDA(NewPos.X,NewPos.Y,FromPos.X,FromPos.Y,
      @DrawAnimatedPixel,LongInt(Designer.Form.Canvas));
    DrawHandle(NewPos);
    LastPos := NewPos;
    LastMask := LineMask;

    LineDrawn := True;
  end;
end;

// -----------------------------------------------------------------------------
procedure TToolbarComponentEditor.SelectControl(Control: TControl; SelectNew: Boolean);
const
  ControlSelected	: Boolean = False;
  OldRect		: TRect = ();

  procedure InvertRect(R: TRect);
  var
    Image		: TBitmap;
  begin
    Image := TBitmap.Create;
    try
      Image.Height := R.Bottom - R.Top;
      Image.Width := R.Right - R.Left;
      Image.Canvas.CopyRect(Rect(0,0,Image.Width,Image.Height), Designer.Form.Canvas, R);
      Designer.Form.Canvas.CopyMode := cmNotSrcCopy;
      Designer.Form.Canvas.CopyRect(R, Image.Canvas, Rect(0,0,Image.Width,Image.Height));
      Designer.Form.Canvas.CopyMode := cmSrcCopy;
    finally
      Image.Free;
    end;
  end;

  procedure FrameRect(R: TRect);
  begin
    Designer.Form.Canvas.DrawFocusRect(R);
    InflateRect(R, -1, -1);
    Designer.Form.Canvas.DrawFocusRect(R);
    InflateRect(R, -1, -1);
    Designer.Form.Canvas.DrawFocusRect(R);
    InflateRect(R, -1, -1);
    Designer.Form.Canvas.DrawFocusRect(R);
  end;

begin
  // Set cursor shape
  if (stCursor in DesignSelectType) then
  begin
    if (SelectNew) then
      Screen.Cursor := crDefault
    else
      Screen.Cursor := crNo;
  end;

  // Remove old selection
  if (ControlSelected) then
  begin
    if (stFrame in DesignSelectType) then
      FrameRect(OldRect);
    if (stInvert in DesignSelectType) then
      InvertRect(OldRect);
    ControlSelected := False;
  end;

  // Draw new selection
  if (SelectNew) and ([stFrame, stInvert] * DesignSelectType <> []) then
  begin
    OldRect.TopLeft := Control.ClientToScreen(Control.ClientRect.TopLeft);
    OldRect.BottomRight := Control.ClientToScreen(Control.ClientRect.BottomRight);
    OldRect.TopLeft := Designer.Form.ScreenToClient(OldRect.TopLeft);
    OldRect.BottomRight := Designer.Form.ScreenToClient(OldRect.BottomRight);
    if (stFrame in DesignSelectType) then
      FrameRect(OldRect);
    if (stInvert in DesignSelectType) then
      InvertRect(OldRect);
    ControlSelected := True;
  end;
end;

// -----------------------------------------------------------------------------
procedure TToolbarComponentEditor.MouseMove(Shift: TShiftState; X, Y: Integer);
var
  TargetControl		: TControl;
  NewPos		: TPoint;

  function ControlAtPos(Parent: TWinControl; const Pos: TPoint; AllowDisabled: Boolean): TControl;
  var
    i			: Integer;
    P			: TPoint;
    Target		: TControl;
  begin
    Result := nil;
    for i := Parent.ControlCount - 1 downto 0 do
    begin
      Result := Parent.Controls[i];
      if not(Result is TWinControl) then
        Continue;
      P := Point(Pos.X - Result.Left, Pos.Y - Result.Top);
      if PtInRect(Result.ClientRect, P) and
        (Result.Perform(CM_HITTEST, 0, Longint(PointToSmallPoint(P))) <> 0) then
      begin
        if (TWinControl(Result).ControlCount > 0) then
        begin
          Target := ControlAtPos(TWinControl(Result),
            Point(Pos.X+Result.Left,Pos.Y+Result.Top) , AllowDisabled);
          if (Target <> nil) then
            Result := Target;
        end;
        Break;
      end;
    end;
    if (Result <> nil) then
      if not(Result is TCustomPanel) then
        Result := nil;
  end;

begin
  NewPos := Point(X,Y);
  ClientToScreen(CaptureWindow, NewPos);
  DrawLine(NewPos, True);
  NewPos := Designer.Form.ScreenToClient(NewPos);

  // Find control under cursor
  TargetControl := ControlAtPos(Designer.Form, NewPos, True);
  if (TargetControl <> LastTarget) then
  begin
    // Repaint previous control and
    // select new control
    SelectControl(TargetControl, (TargetControl <> nil));
    LastTarget := TargetControl;
  end;
end;

// -----------------------------------------------------------------------------
procedure TToolbarComponentEditor.MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
  // Erase old line
  DrawLine(Point(X,Y), False);
  // Remove selection
  SelectControl(nil, False);

  // Restore cursor share
  if (stCursor in DesignSelectType) then
    Screen.Cursor := crDefault;

  // Release cursor
  ClipCursor(nil);

  // Release mouse
  ReleaseCapture;
  IsGrabbing := False;
  if (CaptureWindow <> 0) then
  begin
    DeallocateHWnd(CaptureWindow);
    CaptureWindow := 0;
  end;

  // Link to control
  if (LastTarget <> nil) then
  begin
    TToolbarEditor(Component).Toolbar := (LastTarget as TCustomPanel);
    Designer.Modified;
  end;
end;

// -----------------------------------------------------------------------------
//
//			TToolbarCategoryEditor
//
// -----------------------------------------------------------------------------
function TToolbarCategoryEditor.GetAttributes: TPropertyAttributes;
begin
  Result := [paValueList, paMultiSelect, paSortList];
end;

// -----------------------------------------------------------------------------
procedure TToolbarCategoryEditor.GetValues(Proc: TGetStrProc);
var
  i			: integer;
  item			: TToolbarItem;
begin
  item := GetComponent(0) as TToolbarItem;
  for i := 0 to item.ToolbarEditor.Categories.count-1 do
    Proc(item.ToolbarEditor.Categories[i]);
end;

// -----------------------------------------------------------------------------
//
//			FormDesignToolEdit
//
// -----------------------------------------------------------------------------
procedure TFormDesignToolEdit.FormCreate(Sender: TObject);
begin
  ListDirty := True;
  SortColumn := 2;
end;

// -----------------------------------------------------------------------------
procedure TFormDesignToolEdit.FormDestroy(Sender: TObject);
begin
  Component.DesignTimeDesigner := nil;
end;

// -----------------------------------------------------------------------------
procedure TFormDesignToolEdit.FormShow(Sender: TObject);
begin
  MenuItemImport.enabled := (Component.Menu <> nil);
  BuildList;
  if not(csDesigning in Component.ComponentState) then
    raise Exception.create('Can only use TFormDesignToolEdit at design-time');
end;

// -----------------------------------------------------------------------------
procedure TFormDesignToolEdit.FormClose(Sender: TObject;
  var Action: TCloseAction);
begin
  Action := caFree;
end;

// -----------------------------------------------------------------------------
procedure TFormDesignToolEdit.FormClosed(AForm: TForm);
begin
  if (AForm = Designer.Form) then
    close;
end;

// -----------------------------------------------------------------------------
procedure TFormDesignToolEdit.ComponentDeleted(DelComponent: TComponent);
begin
  if (DelComponent = Component) then
    close;
end;

// -----------------------------------------------------------------------------
procedure TFormDesignToolEdit.FormModified;
begin
  MenuItemImport.enabled := (Component.Menu <> nil);
//  UpdateList;
end;

// -----------------------------------------------------------------------------
procedure TFormDesignToolEdit.Activated;
begin
  UpdateSelection;
end;

// -----------------------------------------------------------------------------
procedure TFormDesignToolEdit.ButtonOKClick(Sender: TObject);
begin
  close;
end;

// -----------------------------------------------------------------------------
procedure TFormDesignToolEdit.UpdatePreview;
var
  Item			: TToolbarItem;

begin
  if (ListViewButtons.ItemFocused = nil) or (ListViewButtons.ItemFocused.Data = nil) then
  begin
    SpeedButtonPreview.Hide;
    LabePreviewlDescription.Hide;
  end else
  begin
    Item := TToolbarItem(ListViewButtons.ItemFocused.Data);
    SpeedButtonPreview.Glyph.assign(Item.Glyph);
    SpeedButtonPreview.NumGlyphs := Item.NumGlyphs;
    SpeedButtonPreview.width := Item.width;
    SpeedButtonPreview.height := Item.height;
    SpeedButtonPreview.hint := Item.hint;
    SpeedButtonPreview.enabled := Item.enabled;
    LabePreviewlDescription.caption := Item.description;
    SpeedButtonPreview.Show;
    LabePreviewlDescription.Show;
  end;
end;

// -----------------------------------------------------------------------------
procedure TFormDesignToolEdit.BuildList;
var
  i			: integer;
  NewItem		: TListItem;
const
  isInside		: Boolean = False;
begin
  if (isInside) then
    exit;
  isInside := True;
  try
    ListDirty := True;
    ListViewButtons.Items.Clear;
    // Add items to list
    for i:=0 to Component.ItemCount-1 do
      if (Component.items[i] <> nil) then
      begin
        NewItem := ListViewButtons.Items.Add;
        NewItem.Data := Component.items[i];
      end;
    // Now add captions and glyph to items
    UpdateList;
  finally
    isInside := False;
  end;
end;

// -----------------------------------------------------------------------------
procedure TFormDesignToolEdit.UpdateList;
var
  i			: integer;
  TempBitmap		,
  OneBitmap		: TBitmap;
const
  isInside		: Boolean = False;
begin
  if (isInside) then
    exit;
  isInside := True;
  try
    oneBitmap := TBitmap.create;
    TempBitmap := TBitmap.create;
    TempBitmap.Height := ImageListButtons.Height;
    TempBitmap.Width := ImageListButtons.Width;
    try
      ImageListButtons.Clear;
      for i:=0 to ListViewButtons.items.count-1 do
      begin
        if (ListViewButtons.items[i].Data = nil) then
        begin
          ListViewButtons.items[i].Caption := '*** Invalid item ***';
          continue;
        end;
        ListViewButtons.items[i].Caption :=
          TToolbarItem(ListViewButtons.items[i].Data).Description;
        ListViewButtons.items[i].SubItems.Clear;
        ListViewButtons.items[i].SubItems.Add(
          TToolbarItem(ListViewButtons.items[i].Data).Category);
        if not(TToolbarItem(ListViewButtons.items[i].Data).Glyph.Empty) then
        begin
          // Get first Glyph if more than one
          if (TToolbarItem(ListViewButtons.items[i].Data).NumGlyphs > 1) then
          begin
            OneBitmap.height := TToolbarItem(ListViewButtons.items[i].Data).Glyph.Height;
            OneBitmap.width := oneBitmap.height;
            OneBitmap.Canvas.CopyRect(Rect(0,0,OneBitmap.Height, OneBitmap.Width),
              TToolbarItem(ListViewButtons.items[i].Data).Glyph.Canvas,
              Rect(0,0,OneBitmap.Height, OneBitmap.Width));
          end else
            OneBitmap.Assign(TToolbarItem(ListViewButtons.items[i].Data).Glyph);

          // Shrink Glyph to fit image list if larger than allowed
          if (OneBitmap.Height > ImageListButtons.Height) or
            (OneBitmap.Width > ImageListButtons.Width) then
            TempBitmap.Canvas.StretchDraw(
              Rect(0,0,ImageListButtons.Width, ImageListButtons.Height),
              OneBitmap)
          else
          // Center Glyph to fit image list if smaller than allowed
          begin
            TempBitmap.Canvas.Brush.Color := OneBitmap.TransparentColor;
            TempBitmap.Canvas.FillRect(Rect(0,0,TempBitmap.Height, TempBitmap.Width));
            TempBitmap.Canvas.Draw((ImageListButtons.Width - OneBitmap.Width) DIV 2,
              (ImageListButtons.Height - OneBitmap.Height) DIV 2, OneBitmap);
          end;
          // Add Glyph to image list
          ListViewButtons.items[i].ImageIndex :=
            ImageListButtons.AddMasked(TempBitmap, TempBitmap.TransparentColor);
        end else
          ListViewButtons.items[i].ImageIndex := -1;
      end;

    finally
      TempBitmap.free;
      OneBitmap.free;
    end;
    SortList;

  finally
    isInside := False;
  end;
end;

// -----------------------------------------------------------------------------
procedure TFormDesignToolEdit.SelectionChanged(ASelection: TComponentList);
begin
  ListDirty := True;
end;

// -----------------------------------------------------------------------------
procedure TFormDesignToolEdit.UpdateSelection;
var
  Components		: TComponentList;
  i			: integer;
begin
  if not(ListDirty) then
    exit;
  ListDirty := False;
  Components := TComponentList.Create;
  try
    for i:= 0 to ListViewButtons.Items.Count-1 do
      if (ListViewButtons.Items[i].Selected) then
        if (ListViewButtons.Items[i].Data <> nil) then
          Components.Add(TComponent(ListViewButtons.Items[i].Data));
    if (Components.Count = 0) then
      Designer.SelectComponent(Component)
    else
      Designer.SetSelections(Components);
    UpdatePreview;
  finally
    Components.Free;
  end;
end;

// -----------------------------------------------------------------------------
procedure TFormDesignToolEdit.ButtonImportClick(Sender: TObject);
var
  FormDesignToolAdd	: TFormDesignToolAdd;
begin
  FormDesignToolAdd := TFormDesignToolAdd.create(self);
  try
    FormDesignToolAdd.Component := Component;
    FormDesignToolAdd.ImportMenuTree;
    if (FormDesignToolAdd.showModal = mrOK) then
      BuildList;
  finally
    FormDesignToolAdd.release;
  end;
end;

// -----------------------------------------------------------------------------
procedure TFormDesignToolEdit.MenuItemRemoveClick(Sender: TObject);
var
  i			: integer;
  idx			: integer;
begin
  if (ListViewButtons.Selected = nil) then
    exit;
  Designer.SelectComponent(Component);
  for i:= 0 to ListViewButtons.Items.Count-1 do
  begin
    if (ListViewButtons.Items[i].Selected) then
    begin
      idx := Component.IndexOf(ListViewButtons.Items[i].Data);
      if (idx >= 0) then
      begin
        ListViewButtons.Items[i].Data := nil;
        Component.Delete(idx);
      end;
    end;
  end;
  BuildList;
end;

// -----------------------------------------------------------------------------
procedure TFormDesignToolEdit.ButtonClearClick(Sender: TObject);
begin
  Designer.SelectComponent(Component);
  Component.Clear;
  BuildList;
end;

// -----------------------------------------------------------------------------
procedure TFormDesignToolEdit.ListViewButtonsChange(Sender: TObject;
  Item: TListItem; Change: TItemChange);
begin
  if (Change = ctState) then
  begin
    ListDirty := True;
    UpdateSelection;
  end;
end;

// -----------------------------------------------------------------------------
procedure TFormDesignToolEdit.ListViewButtonsColumnClick(Sender: TObject;
  Column: TListColumn);
begin
  if (Column.Index+1 = ABS(SortColumn)) then
    // Reverse sort order
    SortColumn := -SortColumn
  else
    SortColumn := Column.Index+1;
  SortList;
end;

// -----------------------------------------------------------------------------
procedure TFormDesignToolEdit.SortList;

  function CustomSortProc(Item1, Item2: TListItem; ParamSort: integer): integer; stdcall;
  var
    AscDesc		: integer;
    s1			,
    s2			: string;
  begin
    if (ParamSort < 0) then
      AscDesc := -1
    else
      AscDesc := 1;
    if (ABS(ParamSort) = 1) then
    begin
      s1 := TListItem(Item1).Caption + TListItem(Item1).SubItems[0];
      s2 := TListItem(Item2).Caption + TListItem(Item2).SubItems[0];
      Result := AscDesc*lstrcmp(PChar(s1), PChar(s2))
    end else
    if (ABS(ParamSort) = 2) then
    begin
      s1 := TListItem(Item1).SubItems[0] + TListItem(Item1).Caption;
      s2 := TListItem(Item2).SubItems[0] + TListItem(Item2).Caption;
      Result := AscDesc*lstrcmp(PChar(s1), PChar(s2))
    end else
      Result := 0;
  end;
begin
  ListViewButtons.CustomSort(@CustomSortProc, SortColumn);
  UpdatePreview;
end;

// -----------------------------------------------------------------------------
procedure TFormDesignToolEdit.ListViewButtonsKeyDown(Sender: TObject;
  var Key: Word; Shift: TShiftState);
begin
  if (Key = VK_INSERT) then
    ButtonImportClick(self)
  else if (Key = VK_DELETE) then
    MenuItemRemoveClick(self);
end;

// -----------------------------------------------------------------------------
procedure TFormDesignToolEdit.FormKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
  case (Key) of
    VK_RETURN:
      begin
        Key := 0;
        ActivateInspector(#0);
      end;
    VK_ESCAPE:
      begin
        Key := 0;
        Close;
      end;
  end;
end;

// -----------------------------------------------------------------------------
procedure TFormDesignToolEdit.MenuItemViewListClick(Sender: TObject);
begin
  MenuItemViewIcons.Checked := False;
  MenuItemViewList.Checked := True;
  ImageListButtons.Height := 16;
  ImageListButtons.Width := 16;
  UpdateList;
  ListViewButtons.ViewStyle := vsReport;
end;

procedure TFormDesignToolEdit.MenuItemViewIconsClick(Sender: TObject);
begin
  MenuItemViewList.Checked := False;
  MenuItemViewIcons.Checked := True;
  ImageListButtons.Height := 25;
  ImageListButtons.Width := 25;
  UpdateList;
  ListViewButtons.ViewStyle := vsIcon;
end;

end.
// CompLib.GetComponentBitmap;

